/*! \file    
**********************************************************************************	
* Title:						Discretix SST API Source file						 					
*																			
* Filename:						sst_api.c															
*																			
* Project, Target, subsystem:	SST Host IF, APIs	
*                               
* Created:						01.07.2007													
*                               
* Modified:																	
*                               
* \Author						Ira Boguslavsky												
*																			
* \Remarks						
*           Copyright (C) 2006 by Discretix Technologies Ltd. All Rights reserved.
**********************************************************************************/

/*----------- External include files ----------------------------------------*/
#include "DX_VOS_Mem.h"
#include "SEPDriver.h"
#include "error.h"
#include "sst_errors.h"
#include "sst_sep_errors.h"
#include "sst_types.h"
#include "sst_host_util.h"
#include "sst_host_def.h"
/*----------- Local Variable declarations -----------------------------------*/
#include "sst_test_flags.h"
#include "sst_host_op_code.h"

/*----------- Local function declarations -----------------------------------*/
/*----------- Local constant definitions  -----------------------------------*/
/*---------------------------------------------------------------------------*/
  
/*---------------------------------------------------------------------------*/
/*               API FUNCTIONS                                               */
/*---------------------------------------------------------------------------*/
 /*Key management services        */

/*SST_AESKeyInsert*/
DxError_t SST_AESKeyInsert(SSTTxnId_t        aTransactionId,
                          SSTSessionId_t     aSessionId,
                          SSTHandle_t        aInitialCtrlAuth,
                          DxByte_t          *aKey_ptr,
                          SSTAESKeyType_t    aKeyType,
                          SSTDataType_t      aDataType,
                          SSTHandle_t       *aKeyHandle_ptr,
                          DxByte_t          *aWorkspace_ptr,
                          DxUint32_t        aWorkspaceSizeInBytes)
{
    DxUint32_t              paramInBuffer_ptr[DX_BYTES_TO_WORD(SST_HOST_IF_API_PARAM_BUFF_KM_AES_INSERT_SIZE_IN_BYTES)];
    DxUint32_t              paramOutBuffer_ptr[DX_BYTES_TO_WORD(SST_HOST_IF_API_RESULT_PARAM_BUFF_KM_AES_INSERT_SIZE_IN_BYTES)];
    DxError_t               errorRC;
    DxUint32_t				keySMPhy;
    DxUint32_t				keySMVirt;
	DxUint32_t				keySize;

    /* Eliminating warnings regarding useless variables */
    aWorkspace_ptr = aWorkspace_ptr;
    aWorkspaceSizeInBytes = aWorkspaceSizeInBytes;

    /* check for null pointers.*/
    if (DX_IS_NULL_2PARAMS(aKey_ptr,aKeyHandle_ptr))
        return SST_RC_ERROR_NULL_POINTER;

	/************************************************************************/
	/*  Lock access to SEP                                                  */
	/************************************************************************/
   errorRC = SEPDriver_Lock();
   if(DX_MNG_SEP_IS_DISABLE_ERR == errorRC)
   {
       errorRC = SST_RC_ERROR_SEP_DISABLED; 
       goto exit_function;
   }
   else if (DX_OK != errorRC)
   {
       errorRC = SST_RC_ERROR_SEP; 
       goto exit_function;
   }
	/************************************************************************/
    /* Prepare input buffer                                                 */
    /************************************************************************/
	/*size and type of key*/
	switch(aKeyType)
	{
	case SST_KEY_TYPE_AES_128_BIT:
	    keySize = 16;
		break;
	case SST_KEY_TYPE_AES_192_BIT:
		keySize = 24;
		break;
	case SST_KEY_TYPE_AES_256_BIT:
		keySize = 32;
		break;
    case SST_KEY_TYPE_AES_512_BIT:
        keySize = 64;
		break;
	default:
		errorRC = SST_RC_ERROR_SEP;
		goto exit_aes_key_insert;
	}
	/*allocate memory in SM*/
	errorRC = SEPDriver_AllocateDataPoolMemory( aKey_ptr, 
                                                keySize, 
                                                &keySMPhy,
                                                &keySMVirt);
	SST_HOST_DRIVER_ERROR_CHECK(errorRC,exit_aes_key_insert) ;
	/*copy password to the SM*/
	errorRC = SEPDriver_WriteToDataPoolMemory(keySMVirt,aKey_ptr, keySize);
	SST_HOST_DRIVER_ERROR_CHECK(errorRC,exit_aes_key_insert) ;
	/* Operation Code */
    paramInBuffer_ptr[0] = SST_SEP_OP_CODE_KM_AES_INSERT;

    /*  Input Parameters */
    paramInBuffer_ptr[1] =  (DxUint32_t)aTransactionId;
    paramInBuffer_ptr[2] =  (DxUint32_t)aSessionId;
    paramInBuffer_ptr[3] =  (DxUint32_t)aInitialCtrlAuth.objDigest;
    paramInBuffer_ptr[4] =  (DxUint32_t)aInitialCtrlAuth.objId;
    paramInBuffer_ptr[5] =  (DxUint32_t)keySMPhy;
    paramInBuffer_ptr[6] =  (DxUint32_t)aKeyType;
    paramInBuffer_ptr[7] =  (DxUint32_t)aDataType; 
    paramInBuffer_ptr[8] =  (DxUint32_t)aKeyHandle_ptr->objDigest;
    paramInBuffer_ptr[9] =  (DxUint32_t)aKeyHandle_ptr->objId;

    /************************************************************************/
    /* Send an input buffer to SeP and wait for output parameters          */
    /************************************************************************/
    errorRC = SST_SendOneMsgGetResponse(paramInBuffer_ptr,
        /* Input buffer length */
        SST_HOST_IF_API_PARAM_BUFF_KM_AES_INSERT_SIZE_IN_BYTES,
        paramOutBuffer_ptr,
        /* Output buffer maximal length */
        SST_HOST_IF_API_RESULT_PARAM_BUFF_KM_AES_INSERT_SIZE_IN_BYTES);
    if (DX_OK != errorRC)
        goto exit_aes_key_insert;

    /************************************************************************/
    /* Analyze return buffer                                                */
    /************************************************************************/
    /* Op code */
    if (paramOutBuffer_ptr[0] != SST_SEP_OP_CODE_KM_AES_INSERT)
    {
        errorRC = SST_RC_ERROR_SEP;
        goto exit_aes_key_insert;
    }
    /* RC */    
	errorRC = paramOutBuffer_ptr[1];
	SST_HOST_RETURN_CODE_CHECK(errorRC,exit_aes_key_insert);

    /* Fill out output parameters */
    aKeyHandle_ptr->objDigest = paramOutBuffer_ptr[2];
    aKeyHandle_ptr->objId     = paramOutBuffer_ptr[3];


exit_aes_key_insert:
	/************************************************************************/
	/* UnLock access to  SEP                                                  */
	/************************************************************************/
	SEPDriver_Unlock();

exit_function:
  
    return errorRC;
}



/*SST_AESKeyGenerate*/
DxError_t SST_AESKeyGenerate(SSTTxnId_t       aTransactionId,
                             SSTSessionId_t   aSessionId,
                             SSTHandle_t      aInitialCtrlAuth,
                             SSTAESKeyType_t  aKeyType,
                             SSTDataType_t    aDataType,
                             SSTHandle_t     *aKeyHandle_ptr,
                             DxByte_t        *aWorkspace_ptr,
                             DxUint32_t       aWorkspaceSizeInBytes)
{
	DxUint32_t              paramInBuffer_ptr [DX_BYTES_TO_WORD(SST_HOST_IF_API_PARAM_BUFF_KM_AES_GEN_SIZE_IN_BYTES)];
	DxUint32_t              paramOutBuffer_ptr[DX_BYTES_TO_WORD(SST_HOST_IF_API_RESULT_PARAM_BUFF_KM_AES_GEN_SIZE_IN_BYTES)];
    DxError_t               errorRC;

    /* Eliminating warnings regarding useless variables */
    aWorkspace_ptr = aWorkspace_ptr;
    aWorkspaceSizeInBytes = aWorkspaceSizeInBytes;

    /* check for null pointers.*/
    if(DX_IS_NULL_PARAM(aKeyHandle_ptr))
        return SST_RC_ERROR_NULL_POINTER;

	/************************************************************************/
	/*  Lock access to SEP                                                  */
	/************************************************************************/
   errorRC = SEPDriver_Lock();
   if(DX_MNG_SEP_IS_DISABLE_ERR == errorRC)
   {
       errorRC = SST_RC_ERROR_SEP_DISABLED; 
       goto exit_function;
   }
   else if (DX_OK != errorRC)
   {
       errorRC = SST_RC_ERROR_SEP; 
       goto exit_function;
   }
	/************************************************************************/
    /* Prepare input buffer                                                 */
    /************************************************************************/
    /* Operation Code */
    paramInBuffer_ptr[0] = SST_SEP_OP_CODE_KM_AES_GEN;

    /*  Input Parameters */
    paramInBuffer_ptr[1] = (DxUint32_t)aTransactionId;
    paramInBuffer_ptr[2] = (DxUint32_t)aSessionId;
    paramInBuffer_ptr[3] = (DxUint32_t)aInitialCtrlAuth.objDigest;
    paramInBuffer_ptr[4] = (DxUint32_t)aInitialCtrlAuth.objId;
    paramInBuffer_ptr[5] = (DxUint32_t)aKeyType;
    paramInBuffer_ptr[6] = (DxUint32_t)aDataType;
    paramInBuffer_ptr[7] = (DxUint32_t)aKeyHandle_ptr->objDigest;
    paramInBuffer_ptr[8] = (DxUint32_t)aKeyHandle_ptr->objId;

    /************************************************************************/
    /* Send an input buffer to SeP and wait for output parameters          */
    /************************************************************************/
    errorRC = SST_SendOneMsgGetResponse(paramInBuffer_ptr,
        /* Input buffer length */
        SST_HOST_IF_API_PARAM_BUFF_KM_AES_GEN_SIZE_IN_BYTES, 
        paramOutBuffer_ptr,
        /* Output buffer length */
        SST_HOST_IF_API_RESULT_PARAM_BUFF_KM_AES_GEN_SIZE_IN_BYTES);  
    if (DX_OK != errorRC)
        goto exit_aes_key_generate;

    /************************************************************************/
    /* Analyze return buffer                                                */
    /************************************************************************/
    /* Op code */
    if (paramOutBuffer_ptr[0] != SST_SEP_OP_CODE_KM_AES_GEN)
    {
        errorRC = SST_RC_ERROR_SEP;
        goto exit_aes_key_generate;
    }

    /* RC */    
	errorRC = paramOutBuffer_ptr[1];
	SST_HOST_RETURN_CODE_CHECK(errorRC,exit_aes_key_generate);

    /* Fill out output parameters */
    aKeyHandle_ptr->objDigest = paramOutBuffer_ptr[2];
    aKeyHandle_ptr->objId     = paramOutBuffer_ptr[3];

exit_aes_key_generate:
	/************************************************************************/
	/* UnLock access to  SEP                                                  */
	/************************************************************************/
	SEPDriver_Unlock();

exit_function:
     
    return errorRC;
}

/*SST_AESKeyRead*/
DxError_t SST_AESKeyRead(SSTSessionId_t     aSessionId,
                         SSTHandle_t        aKeyHandle,
                         DxByte_t          *aKey_ptr,
                         SSTAESKeyType_t   *aKeyType_ptr,
                         DxByte_t          *aWorkspace_ptr,
                         DxUint32_t         aWorkspaceSizeInBytes)
{
	DxUint32_t              paramInBuffer_ptr [DX_BYTES_TO_WORD(SST_HOST_IF_API_PARAM_BUFF_KM_AES_READ_SIZE_IN_BYTES)];
	DxUint32_t              paramOutBuffer_ptr[DX_BYTES_TO_WORD(SST_HOST_IF_API_RESULT_PARAM_BUFF_KM_AES_READ_SIZE_IN_BYTES)];
    DxError_t               errorRC;
    DxError_t               sepErrorRC;
    DxUint32_t				keySMPhyAddress;
    DxUint32_t				keySMVirtAddress;
	DxUint32_t				keySize;
 
    /* Eliminating warnings regarding useless variables */
    aWorkspace_ptr = aWorkspace_ptr;
    aWorkspaceSizeInBytes = aWorkspaceSizeInBytes;
    
    /* Check for NULL pointers.*/
    if (DX_IS_NULL_2PARAMS(aKey_ptr,aKeyType_ptr))
        return SST_RC_ERROR_NULL_POINTER;
	
	/************************************************************************/
	/*  Lock access to SEP                                                  */
	/************************************************************************/
   errorRC = SEPDriver_Lock();
   if(DX_MNG_SEP_IS_DISABLE_ERR == errorRC)
   {
       errorRC = SST_RC_ERROR_SEP_DISABLED; 
       goto exit_function;
   }
   else if (DX_OK != errorRC)
   {
       errorRC = SST_RC_ERROR_SEP; 
       goto exit_function;
   }
	/************************************************************************/
	/* Prepare input buffer                                                 */
	/************************************************************************/
	/*allocate memory in SM*/
	errorRC = SEPDriver_AllocateDataPoolMemory( aKey_ptr, 
                                                SST_HOST_KM_AES_MAX_SIZE_IN_BYTES, 
                                                &keySMPhyAddress,
                                                &keySMVirtAddress);
	SST_HOST_DRIVER_ERROR_CHECK(errorRC,exit_aes_key_read) ;
	
	/* Operation Code */
	paramInBuffer_ptr[0] = SST_SEP_OP_CODE_KM_AES_READ;

    /*  Input Parameters */
    paramInBuffer_ptr[1] = (DxUint32_t)aSessionId;
    paramInBuffer_ptr[2] = (DxUint32_t)aKeyHandle.objDigest;
    paramInBuffer_ptr[3] = (DxUint32_t)aKeyHandle.objId;
    paramInBuffer_ptr[4] = (DxUint32_t)keySMPhyAddress;

    /************************************************************************/
    /* Send an input buffer to SeP and wait for output parameters          */
    /************************************************************************/
    errorRC = SST_SendOneMsgGetResponse(paramInBuffer_ptr,
        /* Input buffer length */
        SST_HOST_IF_API_PARAM_BUFF_KM_AES_READ_SIZE_IN_BYTES,
        paramOutBuffer_ptr,
        /* Output buffer length */
        SST_HOST_IF_API_RESULT_PARAM_BUFF_KM_AES_READ_SIZE_IN_BYTES);
    if (DX_OK != errorRC)
        goto exit_aes_key_read;

    /************************************************************************/
    /* Analyze return buffer                                                */
    /************************************************************************/
    /* Op code */
    if (paramOutBuffer_ptr[0] != SST_SEP_OP_CODE_KM_AES_READ)
    {
        errorRC = SST_RC_ERROR_SEP;
        goto exit_aes_key_read;
    }

    /* RC */    
	errorRC = paramOutBuffer_ptr[1];
	SST_HOST_RETURN_CODE_CHECK(errorRC,exit_aes_key_read);

	/* Fill out output parameters */
	(*aKeyType_ptr) = (SSTAESKeyType_t)paramOutBuffer_ptr[2];

	/*size and type of key*/
	switch(*aKeyType_ptr)
	{
	case SST_KEY_TYPE_AES_128_BIT:
		keySize = 16;
		break;
	case SST_KEY_TYPE_AES_192_BIT:
		keySize = 24;
		break;
	case SST_KEY_TYPE_AES_256_BIT:
		keySize = 32;
		break;
   	case SST_KEY_TYPE_AES_512_BIT:
		keySize = 64;
		break;
	default:
		errorRC = SST_RC_ERROR_SEP;
		goto exit_aes_key_read;
	}

	/*copy from SM*/
	sepErrorRC = SEPDriver_ReadFromDataPoolMemory(keySMVirtAddress,
                                             (aKey_ptr), 
									         keySize);
									   
	SST_HOST_DRIVER_ERROR_CHECK(sepErrorRC,exit_aes_key_read);

exit_aes_key_read:   
	/************************************************************************/
	/* UnLock access to  SEP                                                  */
	/************************************************************************/
	SEPDriver_Unlock();
  
exit_function:
  
    return errorRC;
}

/*SST_DESKeyInsert*/
DxError_t SST_DESKeyInsert(SSTTxnId_t         aTransactionId,
                           SSTSessionId_t     aSessionId,
                           SSTHandle_t        aInitialCtrlAuth,
                           DxByte_t          *aKey_ptr,
                           SSTDESKeyType_t    aKeyType,
                           SSTDataType_t      aDataType,
                           SSTHandle_t       *aKeyHandle_ptr,
                           DxByte_t          *aWorkspace_ptr,
                           DxUint32_t         aWorkspaceSizeInBytes)
{
	DxUint32_t              paramInBuffer_ptr [DX_BYTES_TO_WORD(SST_HOST_IF_API_PARAM_BUFF_KM_DES_INSERT_SIZE_IN_BYTES)];
	DxUint32_t              paramOutBuffer_ptr[DX_BYTES_TO_WORD(SST_HOST_IF_API_RESULT_PARAM_BUFF_KM_DES_INSERT_SIZE_IN_BYTES)];
    DxError_t               errorRC;
	DxUint32_t				keySize;
    DxUint32_t				keySMPhy;
    DxUint32_t				keySMVirt;

    /* Eliminating warnings regarding useless variables */
    aWorkspace_ptr = aWorkspace_ptr;
    aWorkspaceSizeInBytes = aWorkspaceSizeInBytes;

    /* check for null pointers.*/
    if (DX_IS_NULL_2PARAMS(aKey_ptr,aKeyHandle_ptr))
        return SST_RC_ERROR_NULL_POINTER;

	/************************************************************************/
	/*  Lock access to SEP                                                  */
	/************************************************************************/
   errorRC = SEPDriver_Lock();
   if(DX_MNG_SEP_IS_DISABLE_ERR == errorRC)
   {
       errorRC = SST_RC_ERROR_SEP_DISABLED; 
       goto exit_function;
   }
   else if (DX_OK != errorRC)
   {
       errorRC = SST_RC_ERROR_SEP; 
       goto exit_function;
   }
	/************************************************************************/
    /* Prepare input buffer                                                 */
    /************************************************************************/
    /*size and type of key*/
	switch(aKeyType)
	{
	case SST_KEY_TYPE_1DES:
		keySize = 8;
		break;
	case SST_KEY_TYPE_2DES:
		keySize = 16;
		break;
	case SST_KEY_TYPE_3DES:
		keySize = 24;
		break;
	default:
		errorRC = SST_RC_ERROR_SEP;
		goto exit_des_key_insert;
	}
	/*allocate memory in SM*/
	errorRC = SEPDriver_AllocateDataPoolMemory( aKey_ptr, 
                                                keySize, 
                                                &keySMPhy,
                                                &keySMVirt);
	SST_HOST_DRIVER_ERROR_CHECK(errorRC,exit_des_key_insert) ;
	/*copy password to the SM*/
	errorRC = SEPDriver_WriteToDataPoolMemory(keySMVirt, aKey_ptr, keySize);
	SST_HOST_DRIVER_ERROR_CHECK(errorRC,exit_des_key_insert) ;
	/* Operation Code */
    paramInBuffer_ptr[0] = SST_SEP_OP_CODE_KM_DES_INSERT;

    /*  Input Parameters */
    paramInBuffer_ptr[1] =  (DxUint32_t)aTransactionId;
    paramInBuffer_ptr[2] =  (DxUint32_t)aSessionId;
    paramInBuffer_ptr[3] =  (DxUint32_t)aInitialCtrlAuth.objDigest;
    paramInBuffer_ptr[4] =  (DxUint32_t)aInitialCtrlAuth.objId;
    paramInBuffer_ptr[5] =  (DxUint32_t)keySMPhy;
	paramInBuffer_ptr[6] =  (DxUint32_t)aKeyType;
    paramInBuffer_ptr[7] =  (DxUint32_t)aDataType;
    paramInBuffer_ptr[8] =  (DxUint32_t)aKeyHandle_ptr->objDigest;
    paramInBuffer_ptr[9] =  (DxUint32_t)aKeyHandle_ptr->objId;

    /************************************************************************/
    /* Send an input buffer to SeP and wait for output parameters          */
    /************************************************************************/
    errorRC = SST_SendOneMsgGetResponse(paramInBuffer_ptr,
        /* Input buffer length */
        SST_HOST_IF_API_PARAM_BUFF_KM_DES_INSERT_SIZE_IN_BYTES,
        paramOutBuffer_ptr,
        /* Output buffer maximal length */
        SST_HOST_IF_API_RESULT_PARAM_BUFF_KM_DES_INSERT_SIZE_IN_BYTES);
    if (DX_OK != errorRC)
        goto exit_des_key_insert;

    /************************************************************************/
    /* Analyze return buffer                                                */
    /************************************************************************/
    /* Op code */
    if (paramOutBuffer_ptr[0] != SST_SEP_OP_CODE_KM_DES_INSERT)
    {
        errorRC = SST_RC_ERROR_SEP;
        goto exit_des_key_insert;
    }
    /* RC */    
	errorRC = paramOutBuffer_ptr[1];
	SST_HOST_RETURN_CODE_CHECK(errorRC,exit_des_key_insert);

    /* Fill out output parameters */
    aKeyHandle_ptr->objDigest = paramOutBuffer_ptr[2];
    aKeyHandle_ptr->objId     = paramOutBuffer_ptr[3];

exit_des_key_insert: 
	/************************************************************************/
	/* UnLock access to  SEP                                                  */
	/************************************************************************/
	SEPDriver_Unlock();

exit_function:
 
    return errorRC;
}

/*SST_DESKeyGenerate*/
DxError_t SST_DESKeyGenerate(SSTTxnId_t         aTransactionId,
                             SSTSessionId_t     aSessionId,                           
                             SSTHandle_t        aInitialCtrlAuth,   
                             SSTDESKeyType_t    aKeyType,
                             SSTDataType_t      aDataType,
                             SSTHandle_t       *aKeyHandle_ptr,
                             DxByte_t		   *aWorkspace_ptr,
                             DxUint32_t         aWorkspaceSizeInBytes)
{
	DxUint32_t              paramInBuffer_ptr [DX_BYTES_TO_WORD(SST_HOST_IF_API_PARAM_BUFF_KM_DES_GEN_SIZE_IN_BYTES)];
	DxUint32_t              paramOutBuffer_ptr[DX_BYTES_TO_WORD(SST_HOST_IF_API_RESULT_PARAM_BUFF_KM_DES_GEN_SIZE_IN_BYTES)];
    DxError_t               errorRC;

    /* Eliminating warnings regarding useless variables */
    aWorkspace_ptr = aWorkspace_ptr;
    aWorkspaceSizeInBytes = aWorkspaceSizeInBytes;

    /* check for null pointers.*/
    if(DX_IS_NULL_PARAM(aKeyHandle_ptr))
        return SST_RC_ERROR_NULL_POINTER;

	/************************************************************************/
	/*  Lock access to SEP                                                  */
	/************************************************************************/
   errorRC = SEPDriver_Lock();
   if(DX_MNG_SEP_IS_DISABLE_ERR == errorRC)
   {
       errorRC = SST_RC_ERROR_SEP_DISABLED; 
       goto exit_function;
   }
   else if (DX_OK != errorRC)
   {
       errorRC = SST_RC_ERROR_SEP; 
       goto exit_function;
   }
	/************************************************************************/
    /* Prepare input buffer                                                 */
    /************************************************************************/
    /* Operation Code */
    paramInBuffer_ptr[0] = SST_SEP_OP_CODE_KM_DES_GEN;

    /*  Input Parameters */
    paramInBuffer_ptr[1] = (DxUint32_t)aTransactionId;
    paramInBuffer_ptr[2] = (DxUint32_t)aSessionId;
    paramInBuffer_ptr[3] = (DxUint32_t)aInitialCtrlAuth.objDigest;
    paramInBuffer_ptr[4] = (DxUint32_t)aInitialCtrlAuth.objId;
    paramInBuffer_ptr[5] = (DxUint32_t)aKeyType;
    paramInBuffer_ptr[6] = (DxUint32_t)aDataType;
    paramInBuffer_ptr[7] = (DxUint32_t)aKeyHandle_ptr->objDigest;
    paramInBuffer_ptr[8] = (DxUint32_t)aKeyHandle_ptr->objId;

    /************************************************************************/
    /* Send an input buffer to SeP and wait for output parameters          */
    /************************************************************************/
    errorRC = SST_SendOneMsgGetResponse(paramInBuffer_ptr,
        /* Input buffer length */
        SST_HOST_IF_API_PARAM_BUFF_KM_DES_GEN_SIZE_IN_BYTES, 
        paramOutBuffer_ptr,
        /* Output buffer maximal length */
        SST_HOST_IF_API_RESULT_PARAM_BUFF_KM_DES_GEN_SIZE_IN_BYTES);  
    if (DX_OK != errorRC)
        goto exit_des_key_generate;

    /************************************************************************/
    /* Analyze return buffer                                                */
    /************************************************************************/
    /* Op code */
    if (paramOutBuffer_ptr[0] != SST_SEP_OP_CODE_KM_DES_GEN)
    {
        errorRC = SST_RC_ERROR_SEP;
        goto exit_des_key_generate;
    }

    /* RC */    
	errorRC = paramOutBuffer_ptr[1];
	SST_HOST_RETURN_CODE_CHECK(errorRC,exit_des_key_generate);

    /* Fill out output parameters */
    aKeyHandle_ptr->objDigest = paramOutBuffer_ptr[2];
    aKeyHandle_ptr->objId     = paramOutBuffer_ptr[3];

exit_des_key_generate: 
	/************************************************************************/
	/* UnLock access to  SEP                                                  */
	/************************************************************************/
	SEPDriver_Unlock();

exit_function:
    
    return errorRC;
}

/*SST_DESKeyRead*/
DxError_t SST_DESKeyRead(SSTSessionId_t     aSessionId,
                         SSTHandle_t        aKeyHandle,
                         DxByte_t          *aKey_ptr,
                         SSTDESKeyType_t   *aKeyType_ptr,
                         DxByte_t          *aWorkspace_ptr,
                         DxUint32_t         aWorkspaceSizeInBytes)
{
	DxUint32_t              paramInBuffer_ptr [DX_BYTES_TO_WORD(SST_HOST_IF_API_PARAM_BUFF_KM_DES_READ_SIZE_IN_BYTES)];
	DxUint32_t              paramOutBuffer_ptr[DX_BYTES_TO_WORD(SST_HOST_IF_API_RESULT_PARAM_BUFF_KM_DES_READ_SIZE_IN_BYTES)];
    DxError_t               errorRC;
    DxError_t               sepErrorRC;
    DxUint32_t				keySMPhyAddress;
    DxUint32_t				keySMVirtAddress;
	DxUint32_t				keySize;

    /* Eliminating warnings regarding useless variables */
    aWorkspace_ptr = aWorkspace_ptr;
    aWorkspaceSizeInBytes = aWorkspaceSizeInBytes;

    /* check for null pointers.*/
    if (DX_IS_NULL_2PARAMS(aKey_ptr,aKeyType_ptr))
        return SST_RC_ERROR_NULL_POINTER;

	/************************************************************************/
	/*  Lock access to SEP                                                  */
	/************************************************************************/
   errorRC = SEPDriver_Lock();
   if(DX_MNG_SEP_IS_DISABLE_ERR == errorRC)
   {
       errorRC = SST_RC_ERROR_SEP_DISABLED; 
       goto exit_function;
   }
   else if (DX_OK != errorRC)
   {
       errorRC = SST_RC_ERROR_SEP; 
       goto exit_function;
   }
	/************************************************************************/
	/* Prepare input buffer                                                 */
	/************************************************************************/
	/*allocate in SM*/
	errorRC = SEPDriver_AllocateDataPoolMemory( aKey_ptr, 
                                                SST_HOST_KM_DES_MAX_SIZE_IN_BYTES, 
                                                &keySMPhyAddress,
                                                &keySMVirtAddress);
	SST_HOST_DRIVER_ERROR_CHECK(errorRC,exit_des_key_read);
	
	/* Operation Code */
	paramInBuffer_ptr[0] = SST_SEP_OP_CODE_KM_DES_READ;

    /*  Input Parameters */
    paramInBuffer_ptr[1] = (DxUint32_t)aSessionId;
    paramInBuffer_ptr[2] = (DxUint32_t)aKeyHandle.objDigest;
    paramInBuffer_ptr[3] = (DxUint32_t)aKeyHandle.objId;
    paramInBuffer_ptr[4] = (DxUint32_t)keySMPhyAddress;

    /************************************************************************/
    /* Send an input buffer to SeP and wait for output parameters          */
    /************************************************************************/
    errorRC = SST_SendOneMsgGetResponse(paramInBuffer_ptr,
        /* Input buffer length */
        SST_HOST_IF_API_PARAM_BUFF_KM_DES_READ_SIZE_IN_BYTES,
        paramOutBuffer_ptr,
        /* Output buffer maximal length */
        SST_HOST_IF_API_RESULT_PARAM_BUFF_KM_DES_READ_SIZE_IN_BYTES);
    if (DX_OK != errorRC)
        goto exit_des_key_read;

    /************************************************************************/
    /* Analyze return buffer                                                */
    /************************************************************************/
    /* Op code */
    if (paramOutBuffer_ptr[0] != SST_SEP_OP_CODE_KM_DES_READ)
    {
        errorRC = SST_RC_ERROR_SEP;
        goto exit_des_key_read;
    }

    /* RC */    
	errorRC = paramOutBuffer_ptr[1];
	SST_HOST_RETURN_CODE_CHECK(errorRC,exit_des_key_read);

	/* Fill out output parameters */
	(*aKeyType_ptr) = (SSTDESKeyType_t)paramOutBuffer_ptr[2];

	/*size and type of key*/
	switch(*aKeyType_ptr)
	{
	case SST_KEY_TYPE_1DES:
		keySize = 8;
		break;
	case SST_KEY_TYPE_2DES:
		keySize = 16;
		break;
	case SST_KEY_TYPE_3DES:
		keySize = 24;
		break;
	default:
		errorRC = SST_RC_ERROR_SEP;
		goto exit_des_key_read;
	}

	/*copy from SM*/
	sepErrorRC = SEPDriver_ReadFromDataPoolMemory(  keySMVirtAddress,
                                                (aKey_ptr), 
									            keySize);
	SST_HOST_DRIVER_ERROR_CHECK(sepErrorRC,exit_des_key_read);

exit_des_key_read:
	/************************************************************************/
	/* UnLock access to  SEP                                                  */
	/************************************************************************/
	SEPDriver_Unlock();

exit_function:
     
    return errorRC;
}



/*SST_HMACKeyInsert*/
DxError_t SST_HMACKeyInsert(SSTTxnId_t      aTransactionId,                             
                            SSTSessionId_t  aSessionId,
                            SSTHandle_t     aInitialCtrlAuth,   
                            DxByte_t       *aKey_ptr,
                            DxUint32_t      aKeySizeInBytes,
                            SSTDataType_t   aDataType,
                            SSTHandle_t    *aKeyHandle_ptr,
                            DxByte_t       *aWorkspace_ptr,
                            DxUint32_t      aWorkspaceSizeInBytes)
{
	DxUint32_t              paramInBuffer_ptr [DX_BYTES_TO_WORD(SST_HOST_IF_API_PARAM_BUFF_KM_HMAC_INSERT_SIZE_IN_BYTES)];
	DxUint32_t              paramOutBuffer_ptr[DX_BYTES_TO_WORD(SST_HOST_IF_API_RESULT_PARAM_BUFF_KM_HMAC_INSERT_SIZE_IN_BYTES)];
    DxError_t               errorRC;
    DxUint32_t				keySMPhy;
    DxUint32_t				keySMVirt;
  
    /* Eliminating warnings regarding useless variables */
    aWorkspace_ptr = aWorkspace_ptr;
    aWorkspaceSizeInBytes = aWorkspaceSizeInBytes;
  
    /* check for null pointers.*/
    if (DX_IS_NULL_2PARAMS(aKey_ptr,aKeyHandle_ptr))
        return SST_RC_ERROR_NULL_POINTER;

	/************************************************************************/
	/*  Lock access to SEP                                                  */
	/************************************************************************/
   errorRC = SEPDriver_Lock();
   if(DX_MNG_SEP_IS_DISABLE_ERR == errorRC)
   {
       errorRC = SST_RC_ERROR_SEP_DISABLED; 
       goto exit_function;
   }
   else if (DX_OK != errorRC)
   {
       errorRC = SST_RC_ERROR_SEP; 
       goto exit_function;
   }
	/************************************************************************/
    /* Prepare input buffer                                                 */
    /************************************************************************/
	/*allocate memory in SM*/
	errorRC = SEPDriver_AllocateDataPoolMemory( aKey_ptr, 
                                                aKeySizeInBytes, 
                                                &keySMPhy,
                                                &keySMVirt);
	SST_HOST_DRIVER_ERROR_CHECK(errorRC,exit_hmac_key_insert) ;
	/*copy password to the SM*/
	errorRC = SEPDriver_WriteToDataPoolMemory(  keySMVirt, 
                                                aKey_ptr, 
                                                aKeySizeInBytes);
	SST_HOST_DRIVER_ERROR_CHECK(errorRC,exit_hmac_key_insert) ;
	
	/* Operation Code */
    paramInBuffer_ptr[0] = SST_SEP_OP_CODE_KM_HMAC_INSERT;

    /*  Input Parameters */
    paramInBuffer_ptr[1] =  (DxUint32_t)aTransactionId;
    paramInBuffer_ptr[2] =  (DxUint32_t)aSessionId;
    paramInBuffer_ptr[3] =  (DxUint32_t)aInitialCtrlAuth.objDigest;
    paramInBuffer_ptr[4] =  (DxUint32_t)aInitialCtrlAuth.objId;
    paramInBuffer_ptr[5] =  (DxUint32_t)keySMPhy;
    paramInBuffer_ptr[6] =  (DxUint32_t)aKeySizeInBytes;
    paramInBuffer_ptr[7] =  (DxUint32_t)aDataType;
    paramInBuffer_ptr[8] =  (DxUint32_t)aKeyHandle_ptr->objDigest;
    paramInBuffer_ptr[9] =  (DxUint32_t)aKeyHandle_ptr->objId;

    /************************************************************************/
    /* Send an input buffer to SeP and wait for output parameters          */
    /************************************************************************/
    errorRC = SST_SendOneMsgGetResponse(paramInBuffer_ptr,
        /* Input buffer length */
        SST_HOST_IF_API_PARAM_BUFF_KM_HMAC_INSERT_SIZE_IN_BYTES,
        paramOutBuffer_ptr,
        /* Output buffer length */
        SST_HOST_IF_API_RESULT_PARAM_BUFF_KM_HMAC_INSERT_SIZE_IN_BYTES);
    if (DX_OK != errorRC)
        goto exit_hmac_key_insert;

    /************************************************************************/
    /* Analyze return buffer                                                */
    /************************************************************************/
    /* Op code */
    if (paramOutBuffer_ptr[0] != SST_SEP_OP_CODE_KM_HMAC_INSERT)
    {
        errorRC = SST_RC_ERROR_SEP;
        goto exit_hmac_key_insert;
    }
    /* RC */    
	errorRC = paramOutBuffer_ptr[1];
	SST_HOST_RETURN_CODE_CHECK(errorRC,exit_hmac_key_insert);

    /* Fill out output parameters */
    aKeyHandle_ptr->objDigest = paramOutBuffer_ptr[2];
    aKeyHandle_ptr->objId     = paramOutBuffer_ptr[3];

exit_hmac_key_insert:
	/************************************************************************/
	/* UnLock access to  SEP                                                  */
	/************************************************************************/
	SEPDriver_Unlock();
  
exit_function:
  
    return errorRC;
}


/*SST_HMACKeyGenerate*/
DxError_t SST_HMACKeyGenerate(SSTTxnId_t      aTransactionId,
                              SSTSessionId_t  aSessionId,                             
                              SSTHandle_t     aInitialCtrlAuth,   
                              DxUint32_t      aKeySizeInBytes,
                              SSTDataType_t   aDataType,
                              SSTHandle_t    *aKeyHandle_ptr,
                              DxByte_t       *aWorkspace_ptr,
                              DxUint32_t      aWorkspaceSizeInBytes)
{
	DxUint32_t              paramInBuffer_ptr [DX_BYTES_TO_WORD(SST_HOST_IF_API_PARAM_BUFF_KM_HMAC_GEN_SIZE_IN_BYTES)];
	DxUint32_t              paramOutBuffer_ptr[DX_BYTES_TO_WORD(SST_HOST_IF_API_RESULT_PARAM_BUFF_KM_HMAC_GEN_SIZE_IN_BYTES)];
    DxError_t               errorRC;

    /* Eliminating warnings regarding useless variables */
    aWorkspace_ptr = aWorkspace_ptr;
    aWorkspaceSizeInBytes = aWorkspaceSizeInBytes;

    /* check for null pointers.*/
    if(DX_IS_NULL_PARAM(aKeyHandle_ptr))
        return SST_RC_ERROR_NULL_POINTER;

	/************************************************************************/
	/*  Lock access to SEP                                                  */
	/************************************************************************/
   errorRC = SEPDriver_Lock();
   if(DX_MNG_SEP_IS_DISABLE_ERR == errorRC)
   {
       errorRC = SST_RC_ERROR_SEP_DISABLED; 
       goto exit_function;
   }
   else if (DX_OK != errorRC)
   {
       errorRC = SST_RC_ERROR_SEP; 
       goto exit_function;
   }
	/************************************************************************/
    /* Prepare input buffer                                                 */
    /************************************************************************/
    /* Operation Code */
    paramInBuffer_ptr[0] = SST_SEP_OP_CODE_KM_HMAC_GEN;

    /*  Input Parameters */
    paramInBuffer_ptr[1] = (DxUint32_t)aTransactionId;
    paramInBuffer_ptr[2] = (DxUint32_t)aSessionId;
    paramInBuffer_ptr[3] = (DxUint32_t)aInitialCtrlAuth.objDigest;
    paramInBuffer_ptr[4] = (DxUint32_t)aInitialCtrlAuth.objId;
    paramInBuffer_ptr[5] = (DxUint32_t)aKeySizeInBytes;
    paramInBuffer_ptr[6] = (DxUint32_t)aDataType;
    paramInBuffer_ptr[7] = (DxUint32_t)aKeyHandle_ptr->objDigest;
    paramInBuffer_ptr[8] = (DxUint32_t)aKeyHandle_ptr->objId;

    /************************************************************************/
    /* Send an input buffer to SeP and wait for output parameters          */
    /************************************************************************/
    errorRC = SST_SendOneMsgGetResponse(paramInBuffer_ptr,
        /* Input buffer length */
        SST_HOST_IF_API_PARAM_BUFF_KM_HMAC_GEN_SIZE_IN_BYTES, 
        paramOutBuffer_ptr,
        /* Output buffer maximal length */
        SST_HOST_IF_API_RESULT_PARAM_BUFF_KM_HMAC_GEN_SIZE_IN_BYTES);  
    if (DX_OK != errorRC)
        goto exit_hmac_key_generate;

    /************************************************************************/
    /* Analyze return buffer                                                */
    /************************************************************************/
    /* Op code */
    if (paramOutBuffer_ptr[0] != SST_SEP_OP_CODE_KM_HMAC_GEN)
    {
        errorRC = SST_RC_ERROR_SEP;
        goto exit_hmac_key_generate;
    }

    /* RC */    
	errorRC = paramOutBuffer_ptr[1];
	SST_HOST_RETURN_CODE_CHECK(errorRC,exit_hmac_key_generate);

    /* Fill out output parameters */
    aKeyHandle_ptr->objDigest = paramOutBuffer_ptr[2];
    aKeyHandle_ptr->objId     = paramOutBuffer_ptr[3];

exit_hmac_key_generate:
	/************************************************************************/
	/* UnLock access to  SEP                                                  */
	/************************************************************************/
	SEPDriver_Unlock();

exit_function:
     
    return errorRC;
}

/*SST_HMACKeyRead*/
DxError_t SST_HMACKeyRead(SSTSessionId_t  aSessionId,
                          SSTHandle_t     aKeyHandle,
                          DxByte_t       *aKey_ptr,
                          DxUint32_t     *aKeysizeInBytes,
                          DxByte_t       *aWorkspace_ptr,
                          DxUint32_t      aWorkspaceSizeInBytes)
{
	DxUint32_t              paramInBuffer_ptr [DX_BYTES_TO_WORD(SST_HOST_IF_API_PARAM_BUFF_KM_HMAC_READ_SIZE_IN_BYTES)];
	DxUint32_t              paramOutBuffer_ptr[DX_BYTES_TO_WORD(SST_HOST_IF_API_RESULT_PARAM_BUFF_KM_HMAC_READ_SIZE_IN_BYTES)];
    DxError_t               errorRC;
    DxError_t               sepErrorRC;
    DxUint32_t				keySMPhyAddress;
    DxUint32_t				keySMVirtAddress;
 
    /* Eliminating warnings regarding useless variables */
    aWorkspace_ptr = aWorkspace_ptr;
    aWorkspaceSizeInBytes = aWorkspaceSizeInBytes;
 
    /* check for null pointers.*/
    if (DX_IS_NULL_2PARAMS(aKey_ptr,aKeysizeInBytes))
        return SST_RC_ERROR_NULL_POINTER;

	/************************************************************************/
	/*  Lock access to SEP                                                  */
	/************************************************************************/
   errorRC = SEPDriver_Lock();
   if(DX_MNG_SEP_IS_DISABLE_ERR == errorRC)
   {
       errorRC = SST_RC_ERROR_SEP_DISABLED; 
       goto exit_function;
   }
   else if (DX_OK != errorRC)
   {
       errorRC = SST_RC_ERROR_SEP; 
       goto exit_function;
   }
	/************************************************************************/
	/* Prepare input buffer                                                 */
	/************************************************************************/
	/*allocate in SM*/
	errorRC = SEPDriver_AllocateDataPoolMemory( aKey_ptr, 
                                                (*aKeysizeInBytes), 
                                                &keySMPhyAddress,
                                                &keySMVirtAddress);
	SST_HOST_DRIVER_ERROR_CHECK(errorRC,exit_hmac_key_read);
	
	/* Operation Code */
	paramInBuffer_ptr[0] = SST_SEP_OP_CODE_KM_HMAC_READ;

    /*  Input Parameters */
    paramInBuffer_ptr[1] = (DxUint32_t)aSessionId;
    paramInBuffer_ptr[2] = (DxUint32_t)aKeyHandle.objDigest;
    paramInBuffer_ptr[3] = (DxUint32_t)aKeyHandle.objId;
    paramInBuffer_ptr[4] = (DxUint32_t)keySMPhyAddress;

    /************************************************************************/
    /* Send an input buffer to SeP and wait for output parameters           */
    /************************************************************************/
    errorRC = SST_SendOneMsgGetResponse(paramInBuffer_ptr,
        /* Input buffer length */
        SST_HOST_IF_API_PARAM_BUFF_KM_HMAC_READ_SIZE_IN_BYTES,
        paramOutBuffer_ptr,
        /* Output buffer maximal length */
        SST_HOST_IF_API_RESULT_PARAM_BUFF_KM_HMAC_READ_SIZE_IN_BYTES);
    if (DX_OK != errorRC)
        goto exit_hmac_key_read;

    /************************************************************************/
    /* Analyze return buffer                                                */
    /************************************************************************/
    /* Op code */
    if (paramOutBuffer_ptr[0] != SST_SEP_OP_CODE_KM_HMAC_READ)
    {
        errorRC = SST_RC_ERROR_SEP;
        goto exit_hmac_key_read;
    }

    /* RC */    
	errorRC = paramOutBuffer_ptr[1];
	SST_HOST_RETURN_CODE_CHECK(errorRC,exit_hmac_key_read);

	/* Fill out output parameters */
	(*aKeysizeInBytes) = paramOutBuffer_ptr[2];

	/*copy from SM*/
	sepErrorRC = SEPDriver_ReadFromDataPoolMemory(keySMVirtAddress,
                                             (aKey_ptr), 
									         (*aKeysizeInBytes));
	SST_HOST_DRIVER_ERROR_CHECK(sepErrorRC,exit_hmac_key_read);

exit_hmac_key_read:
	/************************************************************************/
	/* UnLock access to  SEP                                                  */
	/************************************************************************/
	SEPDriver_Unlock();
     
exit_function:
     
    return errorRC;
}

/*_DX_SST_RSAKeyInsert*/
DxError_t _DX_SST_RSAKeyInsert(SSTTxnId_t      aTransactionId,
							   SSTSessionId_t  aSessionId,                          
							   SSTHandle_t     aInitialCtrlAuth, 
							   SSTRSAKeyType_t aKeyType,
							   DxUint8_t      *aN_ptr,
							   DxUint32_t      aNsizeInBytes,
							   DxUint8_t      *aD_ptr,
							   DxUint32_t      aDsizeInBytes,
							   DxUint8_t      *aE_ptr, 
							   DxUint32_t      aEsizeInBytes,
							   SSTDataType_t   aDataType,
							   SSTHandle_t    *aKeyHandle_ptr,
							   DxByte_t       *aWorkspace_ptr,
							   DxUint32_t      aWorkspaceSizeInBytes)
{
	DxUint32_t              paramInBuffer_ptr [DX_BYTES_TO_WORD(SST_HOST_IF_API_PARAM_BUFF_KM_RSA_INSERT_SIZE_IN_BYTES)];
	DxUint32_t              paramOutBuffer_ptr[DX_BYTES_TO_WORD(SST_HOST_IF_API_RESULT_PARAM_BUFF_KM_RSA_INSERT_SIZE_IN_BYTES)];
    DxError_t               errorRC;
    DxUint32_t				nSMPhy  = DX_NULL;
    DxUint32_t				eSMPhy  = DX_NULL;
    DxUint32_t				dSMPhy  = DX_NULL;
    DxUint32_t				nSMVirt = DX_NULL;
    DxUint32_t				eSMVirt = DX_NULL;
    DxUint32_t				dSMVirt = DX_NULL;
  
    /* Eliminating warnings regarding useless variables */
    aWorkspace_ptr = aWorkspace_ptr;
    aWorkspaceSizeInBytes = aWorkspaceSizeInBytes;
    
    /* check for null pointers.*/
	switch(aKeyType)
	{
		case SST_KEY_TYPE_RSA_ONLY_PRIVATE_KEY:
			if (DX_IS_NULL_2PARAMS(aN_ptr,aD_ptr))
			{
				return SST_RC_ERROR_NULL_POINTER;
			}
			break;
		case SST_KEY_TYPE_RSA_ONLY_PUBLIC_KEY:
			if (DX_IS_NULL_2PARAMS(aN_ptr,aE_ptr))
			{
				return SST_RC_ERROR_NULL_POINTER;
			}
			break;
		case SST_KEY_TYPE_RSA_BOTH_PUBLIC_AND_PRIVATE_KEY:
			if (DX_IS_NULL_3PARAMS(aN_ptr,aD_ptr,aE_ptr))
			{
				return SST_RC_ERROR_NULL_POINTER;
			}
			break;
		default:
			return SST_RC_FAIL;
	}
	
	if (DX_IS_NULL_PARAM(aKeyHandle_ptr))
	{
		return SST_RC_ERROR_NULL_POINTER;
	}

	/************************************************************************/
	/*  Lock access to SEP                                                  */
	/************************************************************************/
   errorRC = SEPDriver_Lock();
   if(DX_MNG_SEP_IS_DISABLE_ERR == errorRC)
   {
       errorRC = SST_RC_ERROR_SEP_DISABLED; 
       goto exit_function;
   }
   else if (DX_OK != errorRC)
   {
       errorRC = SST_RC_ERROR_SEP; 
       goto exit_function;
   }
	/************************************************************************/
    /* Prepare input buffer                                                 */
    /************************************************************************/
	/*allocate memory in SM*/
	errorRC = SEPDriver_AllocateDataPoolMemory(aN_ptr, aNsizeInBytes, &nSMPhy, &nSMVirt);
	SST_HOST_DRIVER_ERROR_CHECK(errorRC,exit_rsa_key_insert) ;
	/*copy password to the SM*/
	errorRC = SEPDriver_WriteToDataPoolMemory(nSMVirt, aN_ptr, aNsizeInBytes);
	SST_HOST_DRIVER_ERROR_CHECK(errorRC,exit_rsa_key_insert) ;
	/*allocate and copy for aD_ptr if needed*/
	if((SST_KEY_TYPE_RSA_ONLY_PRIVATE_KEY == aKeyType)||
	   (SST_KEY_TYPE_RSA_BOTH_PUBLIC_AND_PRIVATE_KEY == aKeyType))	
	{
		errorRC = SEPDriver_AllocateDataPoolMemory( aD_ptr, 
                                                    aDsizeInBytes, 
                                                    &dSMPhy,
                                                    &dSMVirt);

		SST_HOST_DRIVER_ERROR_CHECK(errorRC,exit_rsa_key_insert) ;
		/*copy password to the SM*/
		errorRC = SEPDriver_WriteToDataPoolMemory( dSMVirt,
                                                   aD_ptr, 
                                                   aDsizeInBytes);
		SST_HOST_DRIVER_ERROR_CHECK(errorRC,exit_rsa_key_insert) ;
	}
	/*allocate and copy for aE_ptr if needed*/
	if ((SST_KEY_TYPE_RSA_ONLY_PUBLIC_KEY == aKeyType)||
		(SST_KEY_TYPE_RSA_BOTH_PUBLIC_AND_PRIVATE_KEY == aKeyType))	
	{
		errorRC = SEPDriver_AllocateDataPoolMemory(aE_ptr, 
                                                   aEsizeInBytes, 
                                                   &eSMPhy,
                                                   &eSMVirt);
		SST_HOST_DRIVER_ERROR_CHECK(errorRC,exit_rsa_key_insert) ;
		/*copy password to the SM*/
		errorRC = SEPDriver_WriteToDataPoolMemory(eSMVirt, aE_ptr, aEsizeInBytes);
		SST_HOST_DRIVER_ERROR_CHECK(errorRC,exit_rsa_key_insert) ;
	}

	/* Operation Code */
    paramInBuffer_ptr[0] = SST_SEP_OP_CODE_KM_RSA_INSERT;

    /*  Input Parameters */
    paramInBuffer_ptr[1] =  (DxUint32_t)aTransactionId;
    paramInBuffer_ptr[2] =  (DxUint32_t)aSessionId;
    paramInBuffer_ptr[3] =  (DxUint32_t)aInitialCtrlAuth.objDigest;
    paramInBuffer_ptr[4] =  (DxUint32_t)aInitialCtrlAuth.objId;
    paramInBuffer_ptr[5] =  (DxUint32_t)aKeyType;
    paramInBuffer_ptr[6] =  (DxUint32_t)nSMPhy;
	paramInBuffer_ptr[7] =  (DxUint32_t)aNsizeInBytes;
	paramInBuffer_ptr[8] =  (DxUint32_t)dSMPhy;
	paramInBuffer_ptr[9] =  (DxUint32_t)aDsizeInBytes;
	paramInBuffer_ptr[10] = (DxUint32_t)eSMPhy;
	paramInBuffer_ptr[11] = (DxUint32_t)aEsizeInBytes;
    paramInBuffer_ptr[12] = (DxUint32_t)aDataType;
    paramInBuffer_ptr[13] = (DxUint32_t)aKeyHandle_ptr->objDigest;
    paramInBuffer_ptr[14] = (DxUint32_t)aKeyHandle_ptr->objId;

    /************************************************************************/
    /* Send an input buffer to SeP and wait for output parameters          */
    /************************************************************************/
    errorRC = SST_SendOneMsgGetResponse(paramInBuffer_ptr,
        /* Input buffer length */
        SST_HOST_IF_API_PARAM_BUFF_KM_RSA_INSERT_SIZE_IN_BYTES,
        paramOutBuffer_ptr,
        /* Output buffer maximal length */
        SST_HOST_IF_API_RESULT_PARAM_BUFF_KM_RSA_INSERT_SIZE_IN_BYTES);
    if (DX_OK != errorRC)
        goto exit_rsa_key_insert;

    /************************************************************************/
    /* Analyze return buffer                                                */
    /************************************************************************/
    /* Op code */
    if (paramOutBuffer_ptr[0] != SST_SEP_OP_CODE_KM_RSA_INSERT)
    {
        errorRC = SST_RC_ERROR_SEP;
        goto exit_rsa_key_insert;
    }
    /* RC */    
	errorRC = paramOutBuffer_ptr[1];
	SST_HOST_RETURN_CODE_CHECK(errorRC,exit_rsa_key_insert);

    /* Fill out output parameters */
    aKeyHandle_ptr->objDigest = paramOutBuffer_ptr[2];
    aKeyHandle_ptr->objId     = paramOutBuffer_ptr[3];

exit_rsa_key_insert: 
	/************************************************************************/
	/* UnLock access to  SEP                                                  */
	/************************************************************************/
	SEPDriver_Unlock();
 
exit_function:

    return errorRC;
}


/*SST_RSAKeyGenerate*/
DxError_t SST_RSAKeyGenerate(SSTTxnId_t      aTransactionId,
							 SSTSessionId_t  aSessionId,
							 SSTHandle_t     aInitialCtrlAuth,  
							 DxUint32_t      aNsizeInBytes,
							 DxByte_t       *aE_ptr,                                     
							 DxUint32_t      aEsizeInBytes,
							 SSTDataType_t   aDataType,
							 SSTHandle_t    *aKeyHandle_ptr,
							 DxByte_t       *aWorkspace_ptr,
							 DxUint32_t      aWorkspaceSizeInBytes)
{
	DxUint32_t              paramInBuffer_ptr [DX_BYTES_TO_WORD(SST_HOST_IF_API_PARAM_BUFF_KM_RSA_GEN_SIZE_IN_BYTES)];
	DxUint32_t              paramOutBuffer_ptr[DX_BYTES_TO_WORD(SST_HOST_IF_API_RESULT_PARAM_BUFF_KM_RSA_GEN_SIZE_IN_BYTES)];
    DxError_t               errorRC;
    DxUint32_t				eSMPhy;
    DxUint32_t				eSMVirt;

    /* Eliminating warnings regarding useless variables */
    aWorkspace_ptr = aWorkspace_ptr;
    aWorkspaceSizeInBytes = aWorkspaceSizeInBytes;

    /* check for null pointers.*/
    if(DX_IS_NULL_2PARAMS(aKeyHandle_ptr,aE_ptr))
        return SST_RC_ERROR_NULL_POINTER;

	/************************************************************************/
	/*  Lock access to SEP                                                  */
	/************************************************************************/
   errorRC = SEPDriver_Lock();
   if(DX_MNG_SEP_IS_DISABLE_ERR == errorRC)
   {
       errorRC = SST_RC_ERROR_SEP_DISABLED; 
       goto exit_function;
   }
   else if (DX_OK != errorRC)
   {
       errorRC = SST_RC_ERROR_SEP; 
       goto exit_function;
   }
	/************************************************************************/
    /* Prepare input buffer                                                 */
    /************************************************************************/
	/*allocate SM*/
	errorRC = SEPDriver_AllocateDataPoolMemory( aE_ptr, 
                                                aEsizeInBytes, 
                                                &eSMPhy,
                                                &eSMVirt);
	SST_HOST_DRIVER_ERROR_CHECK(errorRC,exit_rsa_key_generate) ;
	/*copy password to the SM*/
	errorRC = SEPDriver_WriteToDataPoolMemory(eSMVirt, aE_ptr, aEsizeInBytes);
	SST_HOST_DRIVER_ERROR_CHECK(errorRC,exit_rsa_key_generate) ;
	
	/* Operation Code */
    paramInBuffer_ptr[0] = SST_SEP_OP_CODE_KM_RSA_GEN;

    /*  Input Parameters */
    paramInBuffer_ptr[1] =  (DxUint32_t)aTransactionId;
    paramInBuffer_ptr[2] =  (DxUint32_t)aSessionId;
    paramInBuffer_ptr[3] =  (DxUint32_t)aInitialCtrlAuth.objDigest;
    paramInBuffer_ptr[4] =  (DxUint32_t)aInitialCtrlAuth.objId;
    paramInBuffer_ptr[5] =  (DxUint32_t)aNsizeInBytes;
	paramInBuffer_ptr[6] =  (DxUint32_t)eSMPhy;
	paramInBuffer_ptr[7] =  (DxUint32_t)aEsizeInBytes;
    paramInBuffer_ptr[8] =  (DxUint32_t)aDataType;
    paramInBuffer_ptr[9] =  (DxUint32_t)aKeyHandle_ptr->objDigest;
    paramInBuffer_ptr[10] = (DxUint32_t)aKeyHandle_ptr->objId;

    /************************************************************************/
    /* Send an input buffer to SeP and wait for output parameters          */
    /************************************************************************/
    errorRC = SST_SendOneMsgGetResponse(paramInBuffer_ptr,
        /* Input buffer length */
        SST_HOST_IF_API_PARAM_BUFF_KM_RSA_GEN_SIZE_IN_BYTES, 
        paramOutBuffer_ptr,
        /* Output buffer maximal length */
        SST_HOST_IF_API_RESULT_PARAM_BUFF_KM_RSA_GEN_SIZE_IN_BYTES);  
    if (DX_OK != errorRC)
        goto exit_rsa_key_generate;

    /************************************************************************/
    /* Analyze return buffer                                                */
    /************************************************************************/
    /* Op code */
    if (paramOutBuffer_ptr[0] != SST_SEP_OP_CODE_KM_RSA_GEN)
    {
        errorRC = SST_RC_ERROR_SEP;
        goto exit_rsa_key_generate;
    }

    /* RC */    
	errorRC = paramOutBuffer_ptr[1];
	SST_HOST_RETURN_CODE_CHECK(errorRC,exit_rsa_key_generate);

    /* Fill out output parameters */
    aKeyHandle_ptr->objDigest = paramOutBuffer_ptr[2];
    aKeyHandle_ptr->objId     = paramOutBuffer_ptr[3];

exit_rsa_key_generate: 
	/************************************************************************/
	/* UnLock access to  SEP                                                  */
	/************************************************************************/
	SEPDriver_Unlock();
 
exit_function:
    
    return errorRC;
}


/*_DX_SST_RSAKeyRead*/
DxError_t _DX_SST_RSAKeyRead(SSTSessionId_t   aSessionId,  
							 SSTHandle_t     aKeyHandle, 
							 SSTRSAKeyType_t aKeyType,
							 DxUint8_t      *aN_ptr,
							 DxUint32_t     *aNsizeInBytes_ptr,
							 DxUint8_t      *aD_ptr,
							 DxUint32_t     *aDsizeInBytes_ptr,
							 DxUint8_t      *aE_ptr,
							 DxUint32_t     *aEsizeInBytes_ptr,
							 DxByte_t       *aWorkspace_ptr,
							 DxUint32_t      aWorkspaceSizeInBytes)
{
	DxUint32_t              paramInBuffer_ptr [DX_BYTES_TO_WORD(SST_HOST_IF_API_PARAM_BUFF_KM_RSA_READ_SIZE_IN_BYTES)];
	DxUint32_t              paramOutBuffer_ptr[DX_BYTES_TO_WORD(SST_HOST_IF_API_RESULT_PARAM_BUFF_KM_RSA_READ_SIZE_IN_BYTES)];
    DxError_t               errorRC;
    DxError_t               sepErrorRC;
    DxUint32_t				nSMPhyAddress  = DX_NULL;
    DxUint32_t				nSMVirtAddress = DX_NULL;
    DxUint32_t				eSMPhyAddress  = DX_NULL;
    DxUint32_t				eSMVirtAddress = DX_NULL;
    DxUint32_t				dSMPhyAddress  = DX_NULL;
    DxUint32_t				dSMVirtAddress = DX_NULL;
	DxUint32_t				eSizeVal = 0;
	DxUint32_t				dSizeVal = 0;
 
    /* Eliminating warnings regarding useless variables */
    aWorkspace_ptr = aWorkspace_ptr;
    aWorkspaceSizeInBytes = aWorkspaceSizeInBytes;
 
    /* check for null pointers.*/	
	switch(aKeyType)
	{
		case SST_KEY_TYPE_RSA_ONLY_PRIVATE_KEY:
			if (DX_IS_NULL_2PARAMS(aD_ptr,aDsizeInBytes_ptr))
			{
				return SST_RC_ERROR_NULL_POINTER;
			}
			break;
		case SST_KEY_TYPE_RSA_ONLY_PUBLIC_KEY:
			if (DX_IS_NULL_2PARAMS(aE_ptr,aEsizeInBytes_ptr))
			{
				return SST_RC_ERROR_NULL_POINTER;
			}
		break;
		case SST_KEY_TYPE_RSA_BOTH_PUBLIC_AND_PRIVATE_KEY:
			if (DX_IS_NULL_4PARAMS(aD_ptr,aDsizeInBytes_ptr,aE_ptr,aEsizeInBytes_ptr))
			{
				return SST_RC_ERROR_NULL_POINTER;
			}
			break;
		default:
			return SST_RC_FAIL;
	}
	
	if (DX_IS_NULL_2PARAMS(aN_ptr,aNsizeInBytes_ptr))
	{
		return SST_RC_ERROR_NULL_POINTER;
	}

	/************************************************************************/
	/*  Lock access to SEP                                                  */
	/************************************************************************/
   errorRC = SEPDriver_Lock();
   if(DX_MNG_SEP_IS_DISABLE_ERR == errorRC)
   {
       errorRC = SST_RC_ERROR_SEP_DISABLED; 
       goto exit_function;
   }
   else if (DX_OK != errorRC)
   {
       errorRC = SST_RC_ERROR_SEP; 
       goto exit_function;
   }
	/************************************************************************/
	/* Prepare input buffer                                                 */
	/************************************************************************/
	/*allocate in SM*/
	errorRC = SEPDriver_AllocateDataPoolMemory(aN_ptr, 
                                              (*aNsizeInBytes_ptr), 
                                               &nSMPhyAddress,
                                               &nSMVirtAddress); 
	SST_HOST_DRIVER_ERROR_CHECK(errorRC,exit_rsa_key_read);
	/*allocate for aD_ptr if needed*/
	if((SST_KEY_TYPE_RSA_ONLY_PRIVATE_KEY == aKeyType)||
		(SST_KEY_TYPE_RSA_BOTH_PUBLIC_AND_PRIVATE_KEY == aKeyType))	
	{
		dSizeVal =  (*aDsizeInBytes_ptr);
		errorRC = SEPDriver_AllocateDataPoolMemory( aD_ptr, 
                                                    (*aDsizeInBytes_ptr), 
                                                    &dSMPhyAddress,
                                                    &dSMVirtAddress); 
		SST_HOST_DRIVER_ERROR_CHECK(errorRC,exit_rsa_key_read);
	}
	/*allocate for aE_ptr if needed*/
	if ((SST_KEY_TYPE_RSA_ONLY_PUBLIC_KEY == aKeyType)||
		(SST_KEY_TYPE_RSA_BOTH_PUBLIC_AND_PRIVATE_KEY == aKeyType))	
	{
		eSizeVal = (*aEsizeInBytes_ptr);
		errorRC = SEPDriver_AllocateDataPoolMemory( aE_ptr, 
                                                    (*aEsizeInBytes_ptr), 
                                                    &eSMPhyAddress,
                                                    &eSMVirtAddress);
		SST_HOST_DRIVER_ERROR_CHECK(errorRC,exit_rsa_key_read);
	}

	/* Operation Code */
	paramInBuffer_ptr[0] = SST_SEP_OP_CODE_KM_RSA_READ;

    /*  Input Parameters */
    paramInBuffer_ptr[1] = (DxUint32_t)aSessionId;
    paramInBuffer_ptr[2] = (DxUint32_t)aKeyHandle.objDigest;
    paramInBuffer_ptr[3] = (DxUint32_t)aKeyHandle.objId;
    paramInBuffer_ptr[4] = (DxUint32_t)aKeyType;
	paramInBuffer_ptr[5] = (DxUint32_t)nSMPhyAddress;
	paramInBuffer_ptr[6] = (DxUint32_t)(*aNsizeInBytes_ptr);
	paramInBuffer_ptr[7] = (DxUint32_t)dSMPhyAddress;
	paramInBuffer_ptr[8] = (DxUint32_t)dSizeVal;
	paramInBuffer_ptr[9] = (DxUint32_t)eSMPhyAddress;
	paramInBuffer_ptr[10] = (DxUint32_t)eSizeVal;

    /************************************************************************/
    /* Send an input buffer to SeP and wait for output parameters           */
    /************************************************************************/
    errorRC = SST_SendOneMsgGetResponse(paramInBuffer_ptr,
        /* Input buffer length */
        SST_HOST_IF_API_PARAM_BUFF_KM_RSA_READ_SIZE_IN_BYTES,
        paramOutBuffer_ptr,
        /* Output buffer maximal length */
        SST_HOST_IF_API_RESULT_PARAM_BUFF_KM_RSA_READ_SIZE_IN_BYTES);
    if (DX_OK != errorRC)
        goto exit_rsa_key_read;

    /************************************************************************/
    /* Analyze return buffer                                                */
    /************************************************************************/
    /* Op code */
    if (paramOutBuffer_ptr[0] != SST_SEP_OP_CODE_KM_RSA_READ)
    {
        errorRC = SST_RC_ERROR_SEP;
        goto exit_rsa_key_read;
    }

    /* RC */    
	errorRC = paramOutBuffer_ptr[1];
	SST_HOST_RETURN_CODE_CHECK(errorRC,exit_rsa_key_read);

	/* Fill out output parameters */
	(*aNsizeInBytes_ptr) = paramOutBuffer_ptr[2];
	/*copy from SM*/
	sepErrorRC = SEPDriver_ReadFromDataPoolMemory(  nSMVirtAddress,
                                                (aN_ptr), 
									            (*aNsizeInBytes_ptr));
									  
	SST_HOST_DRIVER_ERROR_CHECK(sepErrorRC,exit_rsa_key_read);

	if (DX_NULL != aDsizeInBytes_ptr)
	{
		(*aDsizeInBytes_ptr) = paramOutBuffer_ptr[3];
		/*copy from SM*/
		sepErrorRC = SEPDriver_ReadFromDataPoolMemory(  dSMVirtAddress,
                                                    (aD_ptr), 
										            (*aDsizeInBytes_ptr));
		SST_HOST_DRIVER_ERROR_CHECK(sepErrorRC,exit_rsa_key_read);
	}
	if (DX_NULL != aEsizeInBytes_ptr)
	{
		(*aEsizeInBytes_ptr) = paramOutBuffer_ptr[4];
		/*copy from SM*/
		sepErrorRC = SEPDriver_ReadFromDataPoolMemory(  eSMVirtAddress,
                                                    (aE_ptr), 
										            (*aEsizeInBytes_ptr));
		SST_HOST_DRIVER_ERROR_CHECK(sepErrorRC,exit_rsa_key_read);
	}

exit_rsa_key_read:  
	/************************************************************************/
	/* UnLock access to  SEP                                                  */
	/************************************************************************/
	SEPDriver_Unlock();

exit_function:
   
    return errorRC;
}

/*_DX_SST_CRTRSAKeyInsert*/
DxError_t _DX_SST_CRTRSAKeyInsert(SSTTxnId_t    aTransactionId,
								  SSTSessionId_t  aSessionId,
								  SSTHandle_t     aInitialCtrlAuth, 
								  SSTRSAKeyType_t aKeyType,
								  DxByte_t        *aN_ptr,
								  DxUint32_t      aNsizeInBytes,
								  DxByte_t       *aE_ptr,
								  DxUint32_t      aEsizeInBytes,
								  DxByte_t       *aP_ptr,
								  DxUint32_t      aPsizeInBytes,
								  DxByte_t       *aQ_ptr,
								  DxUint32_t      aQsizeInBytes,
								  DxByte_t       *aDP_ptr,
								  DxUint32_t      aDPsizeInBytes,
								  DxByte_t       *aDQ_ptr,
								  DxUint32_t      aDQsizeInBytes,
								  DxByte_t       *aQinv_ptr,
								  DxUint32_t      aQinvsizeInBytes,
								  SSTDataType_t   aDataType,
								  SSTHandle_t    *aKeyHandle_ptr,
								  DxByte_t       *aWorkspace_ptr,
								  DxUint32_t      aWorkspaceSizeInBytes)
{
	DxUint32_t              paramInBuffer_ptr [DX_BYTES_TO_WORD(SST_HOST_IF_API_PARAM_BUFF_KM_CRTRSA_INSERT_SIZE_IN_BYTES)];
	DxUint32_t              paramOutBuffer_ptr[DX_BYTES_TO_WORD(SST_HOST_IF_API_RESULT_PARAM_BUFF_KM_CRTRSA_INSERT_SIZE_IN_BYTES)];
    DxError_t               errorRC;
    DxUint32_t				nSMPhy	    = DX_NULL;
    DxUint32_t				eSMPhy	    = DX_NULL;
    DxUint32_t				pSMPhy	    = DX_NULL;
    DxUint32_t				qSMPhy	    = DX_NULL;
    DxUint32_t				dpSMPhy	    = DX_NULL;
    DxUint32_t				dqSMPhy	    = DX_NULL;
    DxUint32_t				qinvSMPhy	= DX_NULL;
    DxUint32_t				nSMVirt     = DX_NULL;
    DxUint32_t				eSMVirt     = DX_NULL;
    DxUint32_t				pSMVirt     = DX_NULL;
    DxUint32_t				qSMVirt     = DX_NULL;
    DxUint32_t				dpSMVirt	= DX_NULL;
    DxUint32_t				dqSMVirt	= DX_NULL;
    DxUint32_t				qinvSMVirt	= DX_NULL;
	
	/* Eliminating warnings regarding useless variables */
    aWorkspace_ptr = aWorkspace_ptr;
    aWorkspaceSizeInBytes = aWorkspaceSizeInBytes;
	
    /* check for null pointers.*/
	switch(aKeyType)
	{
		case SST_KEY_TYPE_RSA_ONLY_PRIVATE_KEY:
			if (DX_IS_NULL_4PARAMS(aP_ptr,aQ_ptr,aDP_ptr,aDQ_ptr)||
				DX_IS_NULL_PARAM(aQinv_ptr))
			{
				return SST_RC_ERROR_NULL_POINTER;
			}
			break;
		case SST_KEY_TYPE_RSA_ONLY_PUBLIC_KEY:
			if (DX_IS_NULL_2PARAMS(aN_ptr,aE_ptr))
			{
				return SST_RC_ERROR_NULL_POINTER;
			}
			break;
		case SST_KEY_TYPE_RSA_BOTH_PUBLIC_AND_PRIVATE_KEY:
			if (DX_IS_NULL_4PARAMS(aP_ptr,aQ_ptr,aDP_ptr,aDQ_ptr)||
				DX_IS_NULL_3PARAMS(aN_ptr,aE_ptr,aQinv_ptr))
			{
				return SST_RC_ERROR_NULL_POINTER;
			}
			break;
		default:
			return SST_RC_FAIL;
	}

	if (DX_IS_NULL_PARAM(aKeyHandle_ptr))
	{
		return SST_RC_ERROR_NULL_POINTER;
	}

	/************************************************************************/
	/*  Lock access to SEP                                                  */
	/************************************************************************/
   errorRC = SEPDriver_Lock();
   if(DX_MNG_SEP_IS_DISABLE_ERR == errorRC)
   {
       errorRC = SST_RC_ERROR_SEP_DISABLED; 
       goto exit_function;
   }
   else if (DX_OK != errorRC)
   {
       errorRC = SST_RC_ERROR_SEP; 
       goto exit_function;
   }
	/************************************************************************/
    /* Prepare input buffer                                                 */
    /************************************************************************/
	switch(aKeyType)
	{
	case SST_KEY_TYPE_RSA_BOTH_PUBLIC_AND_PRIVATE_KEY:
	case SST_KEY_TYPE_RSA_ONLY_PRIVATE_KEY:
		/*P*/
		errorRC = SEPDriver_AllocateDataPoolMemory(aP_ptr, aPsizeInBytes, &pSMPhy, &pSMVirt);
		SST_HOST_DRIVER_ERROR_CHECK(errorRC,exit_crtrsa_key_insert) ;
		errorRC = SEPDriver_WriteToDataPoolMemory(pSMVirt, aP_ptr, aPsizeInBytes);
		SST_HOST_DRIVER_ERROR_CHECK(errorRC,exit_crtrsa_key_insert) ;
		/*Q*/
		errorRC = SEPDriver_AllocateDataPoolMemory(aQ_ptr, aQsizeInBytes, &qSMPhy, &qSMVirt);
		SST_HOST_DRIVER_ERROR_CHECK(errorRC,exit_crtrsa_key_insert) ;
		errorRC = SEPDriver_WriteToDataPoolMemory(qSMVirt, aQ_ptr, aQsizeInBytes);
		SST_HOST_DRIVER_ERROR_CHECK(errorRC,exit_crtrsa_key_insert) ;
		/*DP*/
		errorRC = SEPDriver_AllocateDataPoolMemory(aDP_ptr, aDPsizeInBytes, &dpSMPhy, &dpSMVirt);
		SST_HOST_DRIVER_ERROR_CHECK(errorRC,exit_crtrsa_key_insert) ;
		errorRC = SEPDriver_WriteToDataPoolMemory(dpSMVirt, aDP_ptr, aDPsizeInBytes);
		SST_HOST_DRIVER_ERROR_CHECK(errorRC,exit_crtrsa_key_insert) ;
		/*DQ*/
		errorRC = SEPDriver_AllocateDataPoolMemory(aDQ_ptr, aDQsizeInBytes, &dqSMPhy, &dqSMVirt);
		SST_HOST_DRIVER_ERROR_CHECK(errorRC,exit_crtrsa_key_insert) ;
		errorRC = SEPDriver_WriteToDataPoolMemory(dqSMVirt, aDQ_ptr, aDQsizeInBytes);
		SST_HOST_DRIVER_ERROR_CHECK(errorRC,exit_crtrsa_key_insert) ;
		/*QINV*/
		errorRC = SEPDriver_AllocateDataPoolMemory(aQinv_ptr, aQinvsizeInBytes, &qinvSMPhy, &qinvSMVirt);
		SST_HOST_DRIVER_ERROR_CHECK(errorRC,exit_crtrsa_key_insert) ;
		errorRC = SEPDriver_WriteToDataPoolMemory(qinvSMVirt, aQinv_ptr, aQinvsizeInBytes);
		SST_HOST_DRIVER_ERROR_CHECK(errorRC,exit_crtrsa_key_insert) ;
		if (SST_KEY_TYPE_RSA_ONLY_PRIVATE_KEY == aKeyType)
			break;
		/*Eliminating lint warnings of control flows between cases*/
		/*lint -save -e616*/
	case SST_KEY_TYPE_RSA_ONLY_PUBLIC_KEY:
	    /*lint -restore*/
		/*N*/
		errorRC = SEPDriver_AllocateDataPoolMemory(aN_ptr, aNsizeInBytes, &nSMPhy, &nSMVirt);
		if (DX_SUCCESS != errorRC)
		SST_HOST_DRIVER_ERROR_CHECK(errorRC,exit_crtrsa_key_insert) ;
		errorRC = SEPDriver_WriteToDataPoolMemory(nSMVirt, aN_ptr, aNsizeInBytes);
		SST_HOST_DRIVER_ERROR_CHECK(errorRC,exit_crtrsa_key_insert) ;
		/*E*/
		errorRC = SEPDriver_AllocateDataPoolMemory(aE_ptr, aEsizeInBytes, &eSMPhy, &eSMVirt);
		SST_HOST_DRIVER_ERROR_CHECK(errorRC,exit_crtrsa_key_insert) ;
		errorRC = SEPDriver_WriteToDataPoolMemory(eSMVirt, aE_ptr, aEsizeInBytes);
		SST_HOST_DRIVER_ERROR_CHECK(errorRC,exit_crtrsa_key_insert) ;
		break;
	default:
		return SST_RC_FAIL;
	}
	/* Operation Code */
    paramInBuffer_ptr[0] = SST_SEP_OP_CODE_KM_CRTRSA_INSERT;

    /*  Input Parameters */
    paramInBuffer_ptr[1] =  (DxUint32_t)aTransactionId;
    paramInBuffer_ptr[2] =  (DxUint32_t)aSessionId;
    paramInBuffer_ptr[3] =  (DxUint32_t)aInitialCtrlAuth.objDigest;
    paramInBuffer_ptr[4] =  (DxUint32_t)aInitialCtrlAuth.objId;
    paramInBuffer_ptr[5] =  (DxUint32_t)aKeyType;
    paramInBuffer_ptr[6] =  (DxUint32_t)nSMPhy;
	paramInBuffer_ptr[7] =  (DxUint32_t)aNsizeInBytes;
	paramInBuffer_ptr[8] =  (DxUint32_t)eSMPhy;
	paramInBuffer_ptr[9] =  (DxUint32_t)aEsizeInBytes;
	paramInBuffer_ptr[10] = (DxUint32_t)pSMPhy;
	paramInBuffer_ptr[11] = (DxUint32_t)aPsizeInBytes;
	paramInBuffer_ptr[12] = (DxUint32_t)qSMPhy;
	paramInBuffer_ptr[13] = (DxUint32_t)aQsizeInBytes;
	paramInBuffer_ptr[14] = (DxUint32_t)dpSMPhy;
	paramInBuffer_ptr[15] = (DxUint32_t)aDPsizeInBytes;
	paramInBuffer_ptr[16] = (DxUint32_t)dqSMPhy;
	paramInBuffer_ptr[17] = (DxUint32_t)aDQsizeInBytes;
	paramInBuffer_ptr[18] = (DxUint32_t)qinvSMPhy;
	paramInBuffer_ptr[19] = (DxUint32_t)aQinvsizeInBytes;
    paramInBuffer_ptr[20] = (DxUint32_t)aDataType;
    paramInBuffer_ptr[21] = (DxUint32_t)aKeyHandle_ptr->objDigest;
    paramInBuffer_ptr[22] = (DxUint32_t)aKeyHandle_ptr->objId;

    /************************************************************************/
    /* Send an input buffer to SeP and wait for output parameters          */
    /************************************************************************/
    errorRC = SST_SendOneMsgGetResponse(paramInBuffer_ptr,
        /* Input buffer length */
        SST_HOST_IF_API_PARAM_BUFF_KM_CRTRSA_INSERT_SIZE_IN_BYTES,
        paramOutBuffer_ptr,
        /* Output buffer maximal length */
        SST_HOST_IF_API_RESULT_PARAM_BUFF_KM_CRTRSA_INSERT_SIZE_IN_BYTES);
    if (DX_OK != errorRC)
        goto exit_crtrsa_key_insert;

    /************************************************************************/
    /* Analyze return buffer                                                */
    /************************************************************************/
    /* Op code */
    if (paramOutBuffer_ptr[0] != SST_SEP_OP_CODE_KM_CRTRSA_INSERT)
    {
        errorRC = SST_RC_ERROR_SEP;
        goto exit_crtrsa_key_insert;
    }
    /* RC */    
	errorRC = paramOutBuffer_ptr[1];
	SST_HOST_RETURN_CODE_CHECK(errorRC,exit_crtrsa_key_insert);

    /* Fill out output parameters */
    aKeyHandle_ptr->objDigest = paramOutBuffer_ptr[2];
    aKeyHandle_ptr->objId     = paramOutBuffer_ptr[3];


exit_crtrsa_key_insert: 	
	/************************************************************************/
	/* UnLock access to  SEP                                                  */
	/************************************************************************/
	SEPDriver_Unlock();

exit_function:
 
    return errorRC;
}

/*SST_CRTRSAKeyGenerate*/
DxError_t SST_CRTRSAKeyGenerate(SSTTxnId_t      aTransactionId,
								SSTSessionId_t  aSessionId,
								SSTHandle_t     aInitialCtrlAuth, 
								DxUint32_t      aNsizeInBytes,
								DxByte_t       *aE_ptr,
								DxUint32_t      aEsizeInBytes,
								SSTDataType_t   aDataType,
								SSTHandle_t    *aKeyHandle_ptr,
								DxByte_t       *aWorkspace_ptr,
								DxUint32_t      aWorkspaceSizeInBytes)
{
	DxUint32_t              paramInBuffer_ptr [DX_BYTES_TO_WORD(SST_HOST_IF_API_PARAM_BUFF_KM_CRTRSA_GEN_SIZE_IN_BYTES)];
	DxUint32_t              paramOutBuffer_ptr[DX_BYTES_TO_WORD(SST_HOST_IF_API_RESULT_PARAM_BUFF_KM_CRTRSA_GEN_SIZE_IN_BYTES)];
    DxError_t               errorRC;
    DxUint32_t				eSMPhy;
    DxUint32_t				eSMVirt;

    /* Eliminating warnings regarding useless variables */
    aWorkspace_ptr = aWorkspace_ptr;
    aWorkspaceSizeInBytes = aWorkspaceSizeInBytes;

    /* check for null pointers.*/
    if(DX_IS_NULL_2PARAMS(aKeyHandle_ptr,aE_ptr))
        return SST_RC_ERROR_NULL_POINTER;

	/************************************************************************/
	/*  Lock access to SEP                                                  */
	/************************************************************************/
   errorRC = SEPDriver_Lock();
   if(DX_MNG_SEP_IS_DISABLE_ERR == errorRC)
   {
       errorRC = SST_RC_ERROR_SEP_DISABLED; 
       goto exit_function;
   }
   else if (DX_OK != errorRC)
   {
       errorRC = SST_RC_ERROR_SEP; 
       goto exit_function;
   }
	/************************************************************************/
    /* Prepare input buffer                                                 */
    /************************************************************************/
	/*allocate SM*/
	errorRC = SEPDriver_AllocateDataPoolMemory(aE_ptr, aEsizeInBytes, &eSMPhy, &eSMVirt);
	SST_HOST_DRIVER_ERROR_CHECK(errorRC,exit_crtrsa_key_generate) ;
	/*copy password to the SM*/
	errorRC = SEPDriver_WriteToDataPoolMemory(eSMVirt,aE_ptr, aEsizeInBytes);
	SST_HOST_DRIVER_ERROR_CHECK(errorRC,exit_crtrsa_key_generate) ;
	
	/* Operation Code */
    paramInBuffer_ptr[0] = SST_SEP_OP_CODE_KM_CRTRSA_GEN;

    /*  Input Parameters */
    paramInBuffer_ptr[1] =  (DxUint32_t)aTransactionId;
    paramInBuffer_ptr[2] =  (DxUint32_t)aSessionId;
    paramInBuffer_ptr[3] =  (DxUint32_t)aInitialCtrlAuth.objDigest;
    paramInBuffer_ptr[4] =  (DxUint32_t)aInitialCtrlAuth.objId;
    paramInBuffer_ptr[5] =  (DxUint32_t)aNsizeInBytes;
	paramInBuffer_ptr[6] =  (DxUint32_t)eSMPhy;
	paramInBuffer_ptr[7] =  (DxUint32_t)aEsizeInBytes;
    paramInBuffer_ptr[8] =  (DxUint32_t)aDataType;
    paramInBuffer_ptr[9] =  (DxUint32_t)aKeyHandle_ptr->objDigest;
    paramInBuffer_ptr[10] = (DxUint32_t)aKeyHandle_ptr->objId;

    /************************************************************************/
    /* Send an input buffer to SeP and wait for output parameters          */
    /************************************************************************/
    errorRC = SST_SendOneMsgGetResponse(paramInBuffer_ptr,
        /* Input buffer length */
        SST_HOST_IF_API_PARAM_BUFF_KM_CRTRSA_GEN_SIZE_IN_BYTES, 
        paramOutBuffer_ptr,
        /* Output buffer maximal length */
        SST_HOST_IF_API_RESULT_PARAM_BUFF_KM_CRTRSA_GEN_SIZE_IN_BYTES);  
    if (DX_OK != errorRC)
        goto exit_crtrsa_key_generate;

    /************************************************************************/
    /* Analyze return buffer                                                */
    /************************************************************************/
    /* Op code */
    if (paramOutBuffer_ptr[0] != SST_SEP_OP_CODE_KM_CRTRSA_GEN)
    {
        errorRC = SST_RC_ERROR_SEP;
        goto exit_crtrsa_key_generate;
    }

    /* RC */    
	errorRC = paramOutBuffer_ptr[1];
	SST_HOST_RETURN_CODE_CHECK(errorRC,exit_crtrsa_key_generate);

    /* Fill out output parameters */
    aKeyHandle_ptr->objDigest = paramOutBuffer_ptr[2];
    aKeyHandle_ptr->objId     = paramOutBuffer_ptr[3];

exit_crtrsa_key_generate:
	/************************************************************************/
	/* UnLock access to  SEP                                                  */
	/************************************************************************/
	SEPDriver_Unlock();

exit_function:

    return errorRC;
}


/*_DX_SST_CRTRSAKeyRead*/
DxError_t _DX_SST_CRTRSAKeyRead(SSTSessionId_t  aSessionId,
								SSTHandle_t     aKeyHandle,
								SSTRSAKeyType_t aKeyType,
								DxByte_t       *aN_ptr,
								DxUint32_t     *aNsizeInBytes_ptr,
								DxByte_t       *aE_ptr,
								DxUint32_t     *aEsizeInBytes_ptr,
								DxByte_t       *aP_ptr,
								DxUint32_t     *aPsizeInBytes_ptr,
								DxByte_t       *aQ_ptr,
								DxUint32_t     *aQsizeInBytes_ptr,
								DxByte_t       *aDP_ptr,
								DxUint32_t     *aDPsizeInBytes_ptr,
								DxByte_t       *aDQ_ptr,
								DxUint32_t     *aDQsizeInBytes_ptr,
								DxByte_t       *aQinv_ptr,
								DxUint32_t     *aQinvsizeInBytes_ptr,
								DxByte_t       *aWorkspace_ptr,
								DxUint32_t      aWorkspaceSizeInBytes)
{
	DxUint32_t              paramInBuffer_ptr [DX_BYTES_TO_WORD(SST_HOST_IF_API_PARAM_BUFF_KM_CRTRSA_READ_SIZE_IN_BYTES)];
	DxUint32_t              paramOutBuffer_ptr[DX_BYTES_TO_WORD(SST_HOST_IF_API_RESULT_PARAM_BUFF_KM_CRTRSA_READ_SIZE_IN_BYTES)];
    DxError_t               errorRC;
    DxError_t               sepErrorRC;
    DxUint32_t				nSMPhyAddress	  = DX_NULL;
    DxUint32_t				eSMPhyAddress	  = DX_NULL;
    DxUint32_t				pSMPhyAddress	  = DX_NULL;
    DxUint32_t				qSMPhyAddress	  = DX_NULL;
    DxUint32_t				dpSMPhyAddress    = DX_NULL;
    DxUint32_t				dqSMPhyAddress    = DX_NULL;
    DxUint32_t				qinvSMPhyAddress  = DX_NULL;
    DxUint32_t				nSMVirtAddress	  = DX_NULL;
    DxUint32_t				eSMVirtAddress	  = DX_NULL;
    DxUint32_t				pSMVirtAddress	  = DX_NULL;
    DxUint32_t				qSMVirtAddress	  = DX_NULL;
    DxUint32_t				dpSMVirtAddress   = DX_NULL;
    DxUint32_t				dqSMVirtAddress   = DX_NULL;
    DxUint32_t				qinvSMVirtAddress = DX_NULL;
	DxUint32_t				nSizeVal = 0;
	DxUint32_t				eSizeVal = 0;
	DxUint32_t				pSizeVal = 0;
	DxUint32_t				qSizeVal = 0;
	DxUint32_t				dpSizeVal = 0;
	DxUint32_t				dqSizeVal = 0;
	DxUint32_t				qinvSizeVal = 0;
	
	/* Eliminating warnings regarding useless variables */
    aWorkspace_ptr = aWorkspace_ptr;
    aWorkspaceSizeInBytes = aWorkspaceSizeInBytes;
 
    /* check for null pointers.*/	
	switch(aKeyType)
	{
	case SST_KEY_TYPE_RSA_ONLY_PRIVATE_KEY:
		if (DX_IS_NULL_4PARAMS(aP_ptr,aPsizeInBytes_ptr,aQ_ptr,aQsizeInBytes_ptr) ||
			DX_IS_NULL_4PARAMS(aDP_ptr,aDPsizeInBytes_ptr,aDQ_ptr,aDQsizeInBytes_ptr) ||
			DX_IS_NULL_2PARAMS(aQinv_ptr,aQinvsizeInBytes_ptr))
		{
			return SST_RC_ERROR_NULL_POINTER;
		}
		break;
	case SST_KEY_TYPE_RSA_ONLY_PUBLIC_KEY:
		if (DX_IS_NULL_4PARAMS(aN_ptr,aNsizeInBytes_ptr,aE_ptr,aEsizeInBytes_ptr))
		{
			return SST_RC_ERROR_NULL_POINTER;
		}
		break;
	case SST_KEY_TYPE_RSA_BOTH_PUBLIC_AND_PRIVATE_KEY:
		if (DX_IS_NULL_4PARAMS(aP_ptr,aPsizeInBytes_ptr,aQ_ptr,aQsizeInBytes_ptr)    ||
			DX_IS_NULL_4PARAMS(aDP_ptr,aDPsizeInBytes_ptr,aDQ_ptr,aDQsizeInBytes_ptr)||
			DX_IS_NULL_4PARAMS(aN_ptr,aNsizeInBytes_ptr,aE_ptr,aEsizeInBytes_ptr)    ||
			DX_IS_NULL_2PARAMS(aQinv_ptr,aQinvsizeInBytes_ptr))
		{
			return SST_RC_ERROR_NULL_POINTER;
		}
		break;
	default:
		return SST_RC_FAIL;
	}

	/************************************************************************/
	/*  Lock access to SEP                                                  */
	/************************************************************************/
   errorRC = SEPDriver_Lock();
   if(DX_MNG_SEP_IS_DISABLE_ERR == errorRC)
   {
       errorRC = SST_RC_ERROR_SEP_DISABLED; 
       goto exit_function;
   }
   else if (DX_OK != errorRC)
   {
       errorRC = SST_RC_ERROR_SEP; 
       goto exit_function;
   }
	/************************************************************************/
	/* Prepare input buffer                                                 */
	/************************************************************************/
	switch(aKeyType)
	{
	case SST_KEY_TYPE_RSA_BOTH_PUBLIC_AND_PRIVATE_KEY:
	case SST_KEY_TYPE_RSA_ONLY_PRIVATE_KEY:
		/*P*/
		pSizeVal = *aPsizeInBytes_ptr;
		errorRC = SEPDriver_AllocateDataPoolMemory(aP_ptr, *aPsizeInBytes_ptr, &pSMPhyAddress,&pSMVirtAddress);
		SST_HOST_DRIVER_ERROR_CHECK(errorRC,exit_crtrsa_key_read) ;
		/*Q*/
		qSizeVal = *aQsizeInBytes_ptr;
		errorRC = SEPDriver_AllocateDataPoolMemory(aQ_ptr, *aQsizeInBytes_ptr, &qSMPhyAddress, &qSMVirtAddress);
		SST_HOST_DRIVER_ERROR_CHECK(errorRC,exit_crtrsa_key_read) ;
		/*DP*/
		dpSizeVal = *aDPsizeInBytes_ptr;
		errorRC = SEPDriver_AllocateDataPoolMemory(aDP_ptr, *aDPsizeInBytes_ptr, &dpSMPhyAddress, &dpSMVirtAddress);
		SST_HOST_DRIVER_ERROR_CHECK(errorRC,exit_crtrsa_key_read) ;
		/*DQ*/
		dqSizeVal = *aDQsizeInBytes_ptr;
		errorRC = SEPDriver_AllocateDataPoolMemory(aDQ_ptr, *aDQsizeInBytes_ptr, &dqSMPhyAddress, &dqSMVirtAddress);
		SST_HOST_DRIVER_ERROR_CHECK(errorRC,exit_crtrsa_key_read) ;
		/*QINV*/
		qinvSizeVal = *aQinvsizeInBytes_ptr;
		errorRC = SEPDriver_AllocateDataPoolMemory(aQinv_ptr, *aQinvsizeInBytes_ptr, &qinvSMPhyAddress, &qinvSMVirtAddress);
		SST_HOST_DRIVER_ERROR_CHECK(errorRC,exit_crtrsa_key_read) ;
		if (SST_KEY_TYPE_RSA_ONLY_PRIVATE_KEY == aKeyType)
			break;
		/*Eliminating lint warnings of control flows between cases*/
		/*lint -save -e616*/
	case SST_KEY_TYPE_RSA_ONLY_PUBLIC_KEY:
	    /*lint -restore*/
		/*N*/
		nSizeVal = *aNsizeInBytes_ptr;
		errorRC = SEPDriver_AllocateDataPoolMemory(aN_ptr, *aNsizeInBytes_ptr, &nSMPhyAddress, &nSMVirtAddress);
		if (DX_SUCCESS != errorRC)
		SST_HOST_DRIVER_ERROR_CHECK(errorRC,exit_crtrsa_key_read) ;
		/*E*/
		eSizeVal = *aEsizeInBytes_ptr;
		errorRC = SEPDriver_AllocateDataPoolMemory(aE_ptr, *aEsizeInBytes_ptr, &eSMPhyAddress, &eSMVirtAddress);
		SST_HOST_DRIVER_ERROR_CHECK(errorRC,exit_crtrsa_key_read) ;
		break;
	default:
		return SST_RC_FAIL;
	}
	/* Operation Code */
	paramInBuffer_ptr[0] = SST_SEP_OP_CODE_KM_CRTRSA_READ;

    /*  Input Parameters */
    paramInBuffer_ptr[1] =  (DxUint32_t)aSessionId;
    paramInBuffer_ptr[2] =  (DxUint32_t)aKeyHandle.objDigest;
    paramInBuffer_ptr[3] =  (DxUint32_t)aKeyHandle.objId;
    paramInBuffer_ptr[4] =  (DxUint32_t)aKeyType;
	paramInBuffer_ptr[5] =  (DxUint32_t)nSMPhyAddress;
	paramInBuffer_ptr[6] =  (DxUint32_t)nSizeVal;
	paramInBuffer_ptr[7] =  (DxUint32_t)eSMPhyAddress;
	paramInBuffer_ptr[8] =  (DxUint32_t)eSizeVal;
	paramInBuffer_ptr[9] =  (DxUint32_t)pSMPhyAddress;
	paramInBuffer_ptr[10] = (DxUint32_t)pSizeVal;
	paramInBuffer_ptr[11] = (DxUint32_t)qSMPhyAddress;
	paramInBuffer_ptr[12] = (DxUint32_t)qSizeVal;
	paramInBuffer_ptr[13] = (DxUint32_t)dpSMPhyAddress;
	paramInBuffer_ptr[14] = (DxUint32_t)dpSizeVal;
	paramInBuffer_ptr[15] = (DxUint32_t)dqSMPhyAddress;
	paramInBuffer_ptr[16] = (DxUint32_t)dqSizeVal;
	paramInBuffer_ptr[17] = (DxUint32_t)qinvSMPhyAddress;
	paramInBuffer_ptr[18] = (DxUint32_t)qinvSizeVal;
	
    /************************************************************************/
    /* Send an input buffer to SeP and wait for output parameters           */
    /************************************************************************/
    errorRC = SST_SendOneMsgGetResponse(paramInBuffer_ptr,
        /* Input buffer length */
        SST_HOST_IF_API_PARAM_BUFF_KM_CRTRSA_READ_SIZE_IN_BYTES,
        paramOutBuffer_ptr,
        /* Output buffer maximal length */
        SST_HOST_IF_API_RESULT_PARAM_BUFF_KM_CRTRSA_READ_SIZE_IN_BYTES);
    if (DX_OK != errorRC)
        goto exit_crtrsa_key_read;

    /************************************************************************/
    /* Analyze return buffer                                                */
    /************************************************************************/
    /* Op code */
    if (paramOutBuffer_ptr[0] != SST_SEP_OP_CODE_KM_CRTRSA_READ)
    {
        errorRC = SST_RC_ERROR_SEP;
        goto exit_crtrsa_key_read;
    }

    /* RC */    
	errorRC = paramOutBuffer_ptr[1];
	SST_HOST_RETURN_CODE_CHECK(errorRC,exit_crtrsa_key_read);

	/* Fill out output parameters */
	if (DX_NULL != aNsizeInBytes_ptr)
	{
		(*aNsizeInBytes_ptr) = paramOutBuffer_ptr[2];
		/*copy from SM*/
		sepErrorRC = SEPDriver_ReadFromDataPoolMemory(nSMVirtAddress, 
                                                 (aN_ptr), 
										         (*aNsizeInBytes_ptr)); 
		SST_HOST_DRIVER_ERROR_CHECK(sepErrorRC,exit_crtrsa_key_read);
	}
	if (DX_NULL != aEsizeInBytes_ptr)
	{
		(*aEsizeInBytes_ptr) = paramOutBuffer_ptr[3];
		/*copy from SM*/
		sepErrorRC = SEPDriver_ReadFromDataPoolMemory(eSMVirtAddress,
                                                  (aE_ptr), 
										          (*aEsizeInBytes_ptr)); 

		SST_HOST_DRIVER_ERROR_CHECK(sepErrorRC,exit_crtrsa_key_read);
	}
	if (DX_NULL != aPsizeInBytes_ptr)
	{
		(*aPsizeInBytes_ptr) = paramOutBuffer_ptr[4];
		/*copy from SM*/
		sepErrorRC = SEPDriver_ReadFromDataPoolMemory(  pSMVirtAddress,
                                                    (aP_ptr), 
										            (*aPsizeInBytes_ptr));
		SST_HOST_DRIVER_ERROR_CHECK(sepErrorRC,exit_crtrsa_key_read);
	}
	if (DX_NULL != aQsizeInBytes_ptr)
	{
		(*aQsizeInBytes_ptr) = paramOutBuffer_ptr[5];
		/*copy from SM*/
		sepErrorRC = SEPDriver_ReadFromDataPoolMemory(  qSMVirtAddress,
                                                    (aQ_ptr), 
										            (*aQsizeInBytes_ptr));
		SST_HOST_DRIVER_ERROR_CHECK(sepErrorRC,exit_crtrsa_key_read);
	}
	if (DX_NULL != aDPsizeInBytes_ptr)
	{
		(*aDPsizeInBytes_ptr) = paramOutBuffer_ptr[6];
		/*copy from SM*/
		sepErrorRC = SEPDriver_ReadFromDataPoolMemory(  dpSMVirtAddress,
                                                    (aDP_ptr), 
										            (*aDPsizeInBytes_ptr));
		SST_HOST_DRIVER_ERROR_CHECK(sepErrorRC,exit_crtrsa_key_read);
	}
	if (DX_NULL != aDQsizeInBytes_ptr)
	{
		(*aDQsizeInBytes_ptr) = paramOutBuffer_ptr[7];
		/*copy from SM*/
		sepErrorRC = SEPDriver_ReadFromDataPoolMemory(  dqSMVirtAddress,
                                                    (aDQ_ptr), 
										            (*aDQsizeInBytes_ptr));
		SST_HOST_DRIVER_ERROR_CHECK(sepErrorRC,exit_crtrsa_key_read);
	}
	if (DX_NULL != aQinvsizeInBytes_ptr)
	{
		(*aQinvsizeInBytes_ptr) = paramOutBuffer_ptr[8];
		/*copy from SM*/
		sepErrorRC = SEPDriver_ReadFromDataPoolMemory(  qinvSMVirtAddress,
                                                    (aQinv_ptr), 
										            (*aQinvsizeInBytes_ptr)); 
		SST_HOST_DRIVER_ERROR_CHECK(sepErrorRC,exit_crtrsa_key_read);
	}

exit_crtrsa_key_read: 
	/************************************************************************/
	/* UnLock access to  SEP                                                  */
	/************************************************************************/
	SEPDriver_Unlock();

exit_function:
    
    return errorRC;
}
